#include "timeit.h"

#ifdef WIN32
const double bizilla = 1e9;
const double milli   = 1e-3;
const double nano    = 1e-9;
#else
const double mega    = 1e6;
#endif


//
// getNanoClock - Return value of nanosecond clock as a 64-bit
//                integer (WIN32 only).
//
#ifdef WIN32
__int64 getNanoClock()
{
    LARGE_INTEGER now;
    static LARGE_INTEGER frequency;
    static int gotfrequency = 0;
    __int64 seconds, nsec;

    // Get current value of high-resolution performance counter.
    QueryPerformanceCounter(&now);

    // Get frequency of high-resolution performance counter.
    if (gotfrequency == 0) {
	QueryPerformanceFrequency(&frequency);
	gotfrequency = 1;
    }

    // Calculate seconds by dividing the current value of the
    // high resolution performance counter by the frequency.
    seconds = now.QuadPart / frequency.QuadPart;

    // Calculate number of nanoseconds since the last second
    // by multiplying the number of seconds by the frequency
    // and subtracing the result from the current value of the
    // high-resolution performance counter.  Scale the result
    // divide by the frequency.
    nsec = (bizilla * (now.QuadPart - (seconds * frequency.QuadPart))) / frequency.QuadPart;
    
    // Return the sum of the second and nanoseconds as nanoseconds.
    return( seconds * bizilla + nsec );
}
#endif

//
// timeit_new - Return new timer object.
//
timeobj *   timeit_new()
{
  return( (timeobj *)malloc( sizeof( timeobj ) ) );
}

//
// timeit_delete - Delete timer object.
//
void timeit_delete( timeobj * tt )
{
  free( tt );
}

//
// timeit_start - Return start time in timer object.
//
void timeit_start( timeobj * tt )
{
#ifdef WIN32
  tt->start = getNanoClock();
#else
  gettimeofday( &(tt->start), NULL );
#endif
}

//
// timeit_stop - Return stop time in timer object.
//
void timeit_stop( timeobj * tt )
{
#ifdef WIN32
  tt->stop = getNanoClock();
#else
  gettimeofday( &(tt->stop), NULL );
#endif
}

//
// timeit_geti - Return elapsed time from timer object in seconds 
//               or microseconds as an integer value.
//
int timeit_geti( const timeobj *tt, const timeit_token tok )
{
  int retval = -1;
  switch( tok )
    {
    case timeit_seconds:
      {
	retval = -1;
      }
      break;

    case timeit_useconds:
      {
#ifdef WIN32
	// Microsecond result is nanosecond difference between
        // the stop and start values scaled by 1e-3.
	__int64 tmpval = (__int64)(tt->stop - tt->start);
	tmpval *= milli;
	retval = (int)tmpval;
#else
	// Microsecond result is stop value as microseconds
        // minus start value as microseconds.
	retval = ((tt->stop).tv_sec*mega+(tt->stop).tv_usec) -
	  ((tt->start).tv_sec*mega+(tt->start).tv_usec);
#endif
      }
      break;
    }
  return( retval );
}

//
// timeit_getf - Return elapsed time from timer object in seconds 
//               or microseconds as a float value.
//
float timeit_getf( const timeobj *tt, const timeit_token tok )
{
  float retval = -1.0;
  switch( tok )
    {
    case timeit_seconds:
      {
#ifdef WIN32
	// Result in seconds is nanosecond difference between
        // the stop and start values scaled by 1e-9 to convert
	// the value from nanoseconds to seconds.
	_int64 tmpval = (__int64)(tt->stop - tt->start);
	retval = (float)(tmpval);
	retval *= nano;
#else
	// Result in seconds is microsecond difference between
        // start and stop values converted back to seconds.
        retval = ((tt->stop.tv_sec*mega + (tt->stop).tv_usec) -
	    (tt->start.tv_sec*mega + (tt->start).tv_usec))/mega;
#endif
      }
      break;

    case timeit_useconds:
      {
	retval = -1.0;
      }
      break;
      
    }
  return( retval );
}

