#include <iostream.h>
#include <stdlib.h>
#include <GL/glut.h>

#include <timeit.h>

/////////////////////////////////////////////////////////////////////////////
//
// globals
//
/////////////////////////////////////////////////////////////////////////////

typedef enum motiontype
{
  motion_rotate,
  motion_zoom,
  motion_pan

} motiontype;

typedef struct mouseinfo
{
  GLfloat zoom;
  GLfloat angle[ 2 ];
  GLfloat pan[ 2 ];

  bool moving;
  motiontype type;
  int startx;
  int starty;

} mouseinfo;

mouseinfo mm;
timeobj *fps_timer = timeit_new();
bool silent = false;
bool extra_geometry = false;

/////////////////////////////////////////////////////////////////////////////
//
// adjust the numquads (this is num cubes per side) to increase or
// decrease the workload.
//
/////////////////////////////////////////////////////////////////////////////

const int nobjs = 1000;
const int depth  = 1000;

/////////////////////////////////////////////////////////////////////////////
//
// function prototypes
//
/////////////////////////////////////////////////////////////////////////////

void usage();
void init();
void initproj();
void draw();
void idle();
void reshape( int ww, int hh );
void key( unsigned char key, int x, int y );
void mouse( int button, int xx, int yy );
void motion( int xx, int yy );
void mouseinfo_init( mouseinfo *mm );

void set_high_geometry();
void set_low_geometry();

/////////////////////////////////////////////////////////////////////////////
//
// misc initialization
//
/////////////////////////////////////////////////////////////////////////////

void set_high_geometry()
{
  cerr << "-- enable high geometric load (more triangles)" << endl;
  extra_geometry = true;
}

void set_low_geometry()
{
  cerr << "-- enable low geometric load (fewer triangles)" << endl;
  extra_geometry = false;
}


void reset()
{
  cerr << "-- reset" << endl;
  set_low_geometry();
  silent = false;
  mouseinfo_init( &mm );
  initproj();
}

void mouseinfo_init( mouseinfo *mm )
{
  mm->angle[ 0 ] = 0;
  mm->angle[ 1 ] = 0;
  mm->pan[ 0 ] = 0;
  mm->pan[ 1 ] = 0;
  mm->zoom = 1;
  mm->moving = false;
  mm->type = motion_rotate;
  mm->startx = 0;
  mm->starty = 0;
  
}

void init()
{
  /* gl state */
  glClearColor( .1, 0, .1, 0 );

  glEnable( GL_DEPTH_TEST );
  glDepthFunc( GL_LEQUAL );

  glEnable( GL_LIGHTING );
  glEnable( GL_LIGHT0 );
  float pos[ 4 ] = { 0, 0, 10, 0 };
  glLightfv( GL_LIGHT0, GL_POSITION, pos );
  
  glShadeModel( GL_SMOOTH );

  /* set defaults for program */
  reset();
}

void initproj()
{
  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();

  glMatrixMode( GL_PROJECTION );
  glOrtho( -1.0, 1.0, -1.0, 1.0, -depth, depth );

  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
}

/////////////////////////////////////////////////////////////////////////////
//
// mouse manipulation
//
/////////////////////////////////////////////////////////////////////////////

void mouse( int button, int state, int xx, int yy )
{
  if ( state == GLUT_DOWN )
    {
      mm.moving = true;
      mm.startx = xx;
      mm.starty = yy;
      switch( button )
	{
	case GLUT_LEFT_BUTTON:
	  {
	    mm.type   = motion_rotate;
	  }
	  break;
	case GLUT_MIDDLE_BUTTON:
	  {
	    mm.type   = motion_zoom;
	  }
	  break;
	case GLUT_RIGHT_BUTTON:
	  {
	    mm.type   = motion_pan;
	  }
	  break;
	}
    }
  else
    {
      mm.moving = false;
    }
}

void motion( int xx, int yy )
{
  const float scale = .01;

  if ( mm.moving == true )
    {
      switch( mm.type )
	{
	case motion_rotate:
	  {
	    mm.angle[ 0 ] += xx - mm.startx;
	    mm.angle[ 1 ] += yy - mm.starty;
	  }
	  break;

	case motion_pan:
	  {
	    mm.pan[ 0 ] += scale * ( xx - mm.startx );
	    mm.pan[ 1 ] += scale * ( yy - mm.starty );
	  }
	  break;

	case motion_zoom:
	  {
	    mm.zoom *= 1.0 + ( scale * ( yy - mm.starty ) );

	  }
	  break;
	}
      
      mm.startx = xx;
      mm.starty = yy;

      glutPostRedisplay();
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// drawing / resizing functions
//
/////////////////////////////////////////////////////////////////////////////

void reshape( int ww, int hh )
{
  glViewport( 0, 0, ww, hh );

  initproj();
}

void draw()
{
  /* clear screen */
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

  /* set eyepoint */
  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();

  glTranslatef( 0, 0, -1 );
  glTranslatef( mm.pan[ 0 ], -mm.pan[ 1 ], 0.0 );
  glRotatef( mm.angle[ 1 ], 1.0, 0.0, 0.0 );
  glRotatef( mm.angle[ 0 ], 0.0, 1.0, 0.0 );

  /* change to a translate */
  glScalef( mm.zoom, mm.zoom, mm.zoom );
  
  /* set default color */
  GLfloat green[ 4 ] = { 0, 1, 0, 0 };
  glMaterialfv( GL_FRONT, GL_DIFFUSE, green );

  /* set a bunch of defaults for the geometry */
  const float scale = 1.0;
  const float scale2 = scale/2.0;
  const float step = depth/nobjs;
  const int   ncubes = nobjs;

  /* draw our regular geometry */
  glTranslatef( 0, 0, depth );
  glNormal3f( 0, 0, 1 );
  timeit_start( fps_timer );
  for( int ii=0; ii<nobjs; ii++ )
    {
      glTranslatef( 0, 0, -step );
      glBegin( GL_TRIANGLES );
      glVertex3f( -scale2, -scale2, 0 );
      glVertex3f(  scale2, -scale2, 0 );
      glVertex3f(  scale2,  scale2, 0 );
      glEnd();
    }

  /* draw extra geometry if selected */
  if ( extra_geometry )
    {
      glTranslatef( scale2, scale2, depth );
      for ( ii=0; ii<ncubes; ii++ )
	{
	  glTranslatef( 0, 0, -step );
	  glutSolidCube( scale2 / 4.0 );
	}
    }

  /* dump a fps each frame we draw */
  if ( !silent )
    {
      glFinish();
      timeit_stop( fps_timer );
      cerr << "-- fps: " << 1.0 /
	timeit_getf( fps_timer, timeit_seconds ) << endl;
    }

  /* if we used a doublebuffered visual, we'd need a swap here */
}


/////////////////////////////////////////////////////////////////////////////
//
// key-event handler
//
/////////////////////////////////////////////////////////////////////////////

void usage()
{
  cerr << "-- mouse commands" << endl;
  cerr << "l - rotate the scene" << endl;
  cerr << "m - zoom (scale) the scene" << endl;
  cerr << "r - pan the scene" << endl;
  cerr << "-- keyboard commands" << endl;
  cerr << "r - reset all parameters" << endl;
  cerr << "G - set high geometry load" << endl;
  cerr << "g - set low geometry load" << endl;
  cerr << "t - run timing test to generate triangles-per-second" << endl;
  cerr << "h - display this help" << endl;
  cerr << "? - display this help" << endl;
}

void key( unsigned char key, int, int )
{
  bool need_refresh = false;

  switch( key )
    {
    case 'h':
    case '?':
      {
	usage();
      }
      break;
      
    case 'r':
      {
	reset();
	need_refresh = true;
      }
      break;
      
    case 'G':
      { 
	set_high_geometry();
	need_refresh = true;
      }
      break;

    case 'g':
      { 
	set_low_geometry();
	need_refresh = true;
      }
      break;

    case 't':
      {
	cerr << "-- running test" << endl;
	reset();
	float time = -1;
	float tps = 0;
	int ii;
	const int niters = 10;
	silent = true;

	timeobj *tt = timeit_new();

	/* run test of high geometric load, nominal fill load increase */
	set_high_geometry();
	timeit_start( tt );
	for( ii=0; ii<niters; ii++ )
	  {
	    draw();
	  }
	timeit_stop( tt );
	time = timeit_getf( tt, timeit_seconds );
	tps = ( nobjs + ( extra_geometry ? ( nobjs*6*2 ) : 0) )
	  * niters / time;

	cerr << "-- test results w/high geometric load" << endl;
	cerr << "total time (s)       : " << time << endl;
	cerr << "triangles per second : " << tps << endl;

	/* run test of low geometric load, baseline fill load */
	set_low_geometry();
	timeit_start( tt );
	for( ii=0; ii<niters; ii++ )
	  {
	    draw();
	  }
	timeit_stop( tt );
	time = timeit_getf( tt, timeit_seconds );
	tps = ( nobjs + ( extra_geometry ? ( nobjs*6*2 ) : 0) )
	  * niters / time;

	cerr << "-- test results w/low geometric load" << endl;
	cerr << "total time (s)       : " << time << endl;
	cerr << "triangles per second : " << tps << endl;

	/* cleanup */
	timeit_delete( tt );
	need_refresh = false;
	silent = false;
      }
      break;
      
    case 'q':
    case 27:
      {
	exit(0);
      }
      break;
    }

  if ( need_refresh == true )
    {
      draw();
    }
}

/////////////////////////////////////////////////////////////////////////////
//
// main (setup, visual selection, callback registration)
//
/////////////////////////////////////////////////////////////////////////////

int main(int argc, char **argv)
{
  glutInit( &argc, argv );
  
  /* note: all drawing is done in single-buffer mode to avoid frame
   * quantization issues as described in the course notes.
   */

  glutInitDisplayMode( GLUT_RGB | GLUT_SINGLE | GLUT_DEPTH );
  glutInitWindowSize( 500, 500 );
  glutCreateWindow( "glut test shell" );
  
  init();
  
  glutReshapeFunc( reshape );
  glutKeyboardFunc( key );
  glutMouseFunc( mouse );
  glutMotionFunc( motion );
  glutDisplayFunc( draw );
  glutMainLoop();

  return( 1 );
}
