unit Sgraph ;

  {

   Module  : Graphic routines IBM PC

   Author  : Hans Otten

   Version : 2.1  21-oct-1991

   Facility: Convert PCX, SIXEL and MSX picture files

   Purpose : offer graphic facilities in

             low   resolution
             high  resolution
             auto  resolution

    Definition of low or high depends on graphics device.


  }

interface

  uses

    CNV ;


  procedure CheckDisplay ;

  procedure Graph_Init (var parse_info : parse_infotype ;
                        colormap : colormap_type ;
                        display_info : display_infotype ;
                        vdisplay : vdisplay_type) ;

  procedure Display_pixel (x,y : integer ; color: byte ) ;

  procedure Close_Graph ;


implementation

  {

    PC resolutions:

    Hercules
        low  = 768x576   mono
        high = 768x576   mono

    CGA
        low  = 320x200   4 colors
        high = 640x200   4 colors

    EGA
        low  = 320x200   16 colors
        high = 640x480   16 colors

    VGA
        low  = 320x200 256 colors
        high = 640x480  16 colors

    Super-VGA
       low   = 320x200 256 colors
       high  = 640x400 256 colors
               640x480 256 colors
               800x600 256 colors
              1024x768 256 colors

  }

  uses

    DOS, CRT, GRAPH,
    SBUFFER ;


  type

    ColorValue     = record
                       Rv, Gv, Bv: 0..63 ;
                     end;
    VGADACType = array[0..255] of ColorValue;
    Palettetype  = array[0..16] of byte ;

    PC_devs_type = (pc_herc, pc_CGA, pc_EGA, pc_VGA, pc_SVGA, pc_none) ;

  var

    DAC : VGADACtype ;
    GraphDriver, GraphMode : integer;
    ErrorCode : integer;
    AutoDetectPointer : pointer;
    remap : array[0..255] of byte ;
    offset : word ;
    oldval : byte ;

    { variables used to locate/load BGI driver }
    S_low     : PathStr;
    Dir_low   : DirStr;
    Name_low  : NameStr;
    Ext_low   : ExtStr;
    S_high    : PathStr;
    Dir_high  : DirStr;
    Name_high : NameStr;
    Ext_high  : ExtStr;

    { returned by CheckDisplay }
    GraphmodeHigh,
    maxGraphmodeHigh,
    Graphmodelow   : integer ;
    BGI_name_low,
    BGI_name_high  : PathStr ;
    pc_device : PC_devs_type ;
    { sizes of display low and high }
    Xmaxlow,  Ymaxlow,
    XmaxHigh, Ymaxhigh : integer ;
    max_colors_low,
    max_colors_high : integer ;
    displaytype : vdisplay_type ;

  {$F+}
  function DetectVGA256 : integer;

    var

     Suggestedmode,
     DetectedDriver : integer;

  begin

    DetectGraph(DetectedDriver, SuggestedMode);
    if (DetectedDriver = VGA) or (DetectedDriver = MCGA)
      then
        begin
          if displaytype = low
            then
              DetectVGA256 := 0
            else
              DetectVGA256 := GraphModeHigh
        end
      else
        DetectVGA256 := grError; { Couldn't detect hardware }

   end; { DetectVGA256 }
  {$F-}

procedure FillVGADAC ;

  {
   fill MCGA or VGA DAC registers with RGB values
  }

  var

    Regs : Registers;

  begin

    with Regs do
      begin
        AX := $1012;
        BX := 0;
        CX := 255 ;
        ES := Seg(DAC);
        DX := Ofs(DAC);
      end;
    Intr($10, Regs);

  end; { FillVGADAC }

procedure SetPalette(var Palette : PaletteType );

  {
    fill EGA palette registers
    for VGA palette entry is index into DAC
    for EGA it is RGB info
  }

  var

    Regs : Registers;

  begin

    with Regs do
      begin
        AX := $1002;
        ES := Seg(Palette);
        DX := Ofs(Palette);
      end;
    Intr($10, Regs);

  end; { SetPalette }

  procedure InitVGAPalette256(vdisplay : vdisplay_type ;
                              used_colors: integer ;
                              colormap : colormap_type) ;

    { Initialise VGA palette }

    var

      palette : palettetype ;
      vr,vg,vb  : byte ;
      nr_of_colors,
      color_count,
      remap_count : integer ;

    procedure FillDAC(color_nr, r,g,b : integer) ;

       begin

         DAC[color_nr].Rv := r ;
         DAC[color_nr].Gv := g ;
         DAC[color_nr].Bv := b ;

       end ; { FillDAC }

    begin

      if vdisplay = high
        then
          nr_of_colors := max_colors_high
        else
          nr_of_colors := max_colors_low ;
      remap_count := 0 ;
      { try to reserve color 0 for background }
      if used_colors < nr_of_colors
        then
          begin
            inc(remap_count) ;
            FillDAC(0,0,0,0) ;
          end ;
      for color_count := 0 to max_color do
        begin
          remap[color_count] := 0 ;
          if colormap[color_count].used
            then
              begin
                remap[color_count] := remap_count ;
                vr := colormap[color_count].colors[c_red] ;
                vr := (vr * 63) div colormax ;
                vg := colormap[color_count].colors[c_green] ;
                vg := (vg * 63) div colormax ;
                vb := colormap[color_count].colors[c_blue] ;
                vb := (vb * 63) div colormax ;
                FillDAC(remap_count,vr,vg,vb) ;
                inc(remap_count) ;
              end ;
        end ;

      FillVGADAC ;

      { fill color palette registers to point to DAC registers 0..15 }
      Palette[16] := 0 ;
      for color_count := 0 to 15 do
        Palette[color_count] := color_count ;
      SetPalette(Palette) ;

    end ; { VGAPalette256 }

  procedure BWpalette (colormap : colormap_type)  ;

    { initialise remapping for other than VGA PC devices }

    { In this version only black and white supported
      Could be improved for EGA palette
    }

    var

      remap_count,
      color_count : integer ;

    begin

      remap_count := 0 ;
      for color_count := 0 to max_color do
        begin
          remap[color_count] := 0 ;
          if colormap[color_count].used
            then
              begin
                remap[color_count] := 0 ;
                { check if intense enough to become white }
                if (colormap[color_count].colors[c_red]   +
                    colormap[color_count].colors[c_green] +
                    colormap[color_count].colors[c_blue] )  > 240
                  then
                    remap[color_count] := 1 ;
                inc(remap_count) ;
              end ;
        end ;
      { force auto display to use high-resolution }
      max_colors_high := 256 ;
      max_colors_low  := 256 ;

    end ; { BWPalette }


  procedure InitVGA256(BGI_name : string ;
                       Dir : Dirstr;
                       Graphmode : integer) ;

    { initialise VGA into a 256 colors colormode
      BGI_name is name of BGI driver (VGA256 or SVGA256) }

    begin

      ClrScr;
      DirectVideo := false;
      AutoDetectPointer := @DetectVGA256; { Point to detection routine }
      GraphDriver := InstallUserDriver(BGI_name, AutoDetectPointer);
      GraphDriver := Detect;
      InitGraph(GraphDriver, GraphMode, Dir);
      ErrorCode := GraphResult;
      if ErrorCode <> grOK then
        begin
          CloseGraph ;
          CloseFiles ;
          Writeln('Graphic systemerror: ', GraphErrorMsg(ErrorCode));
          Halt(1);
        end;
      setcolor(15) ;

    end ; { InitVGA256 }


  procedure VGraphInit(dir : dirstr ; graphmode : integer) ;

    { initialise standard graphics in detected type (CheckDisplay)
      and in desired mode }

    var

      GraphDriver, t_graphmode : integer ;

    begin

      ClrScr;
      DirectVideo := false;
      DetectGraph(GraphDriver, T_Graphmode) ;
      InitGraph(GraphDriver, GraphMode, Dir);
      ErrorCode := GraphResult;
      if ErrorCode <> grOK then
        begin
          CloseGraph ;
          CloseFiles ;
          Writeln('Graphic system error: ', GraphErrorMsg(ErrorCode));
          Halt(1);
        end;

    end ; { VGraphInit }


  procedure Graph_Init (var parse_info : parse_infotype ;
                        colormap : colormap_type ;
                        display_info : display_infotype ;
                        vdisplay : vdisplay_type) ;


      {
        initialize graphics dependent on
        nr of colors and size of pictures
        and capabilities of graphic device
      }

      begin

        { if auto resolution determine best mode }
        if vdisplay = auto
          then
            begin
              with display_info do
                begin
                  if ( ((end_view_row - start_view_row) < Ymaxlow) and
                       ((end_view_col - start_view_col) < XmaxLow)     ) or
                       (parse_info.used_colors > max_colors_high)
                    then
                      vdisplay := low
                    else
                      vdisplay := high
                end ;
            end ;

        displaytype := vdisplay ;

        if vdisplay = high
          then
            begin
              { Super VGA requires special user init routine }
              if  pc_device = pc_SVGA
                then
                  begin
                    { find lowest high resolution that fits
                      on screen }
                    with display_info do
                      begin
                        if ((end_view_row - start_view_row) < 400 ) and
                           ((end_view_col - start_view_col) < 600)
                           then
                             GraphmodeHigh := 1
                        else if ((end_view_row - start_view_row) < 480) and
                                ((end_view_col - start_view_col) < 600)
                           then
                             GraphmodeHigh := 2
                        else if ((end_view_row - start_view_row) < 600) and
                                ((end_view_col - start_view_col) < 800 )
                           then
                             GraphmodeHigh := 3
                        else { use the 1024x768 mode }
                          GraphmodeHigh := 4
                      end ;
                    if GraphmodeHigh > maxGraphmodeHigh
                      then
                        GraphmodeHigh := maxGraphmodeHigh ;
                    InitVGA256(name_high, dir_high, GraphmodeHigh) ;
                  end
                else
                  VgraphInit(dir_high, GraphmodeHigh)
            end
        else { low resolution }
          begin
            if (pc_device = pc_VGA) or (pc_device = pc_SVGA)
              then
                InitVGA256(name_low, dir_low, graphmodelow)
              else
                VgraphInit(dir_low, Graphmodelow)
          end ;

      if (pc_device = pc_VGA) or (pc_device = pc_SVGA)
        then
          InitVGAPalette256(vdisplay,
                            parse_info.used_colors,
                            colormap)
        else
          BWPalette(colormap) ;

    end ; { Graph_init }

  procedure Display_pixel (x,y : integer ; color: byte ) ;

    begin

      dec(x) ;
      dec(y) ;
      if (PC_device = pc_VGA) or (pc_device = pc_SVGA)
        then
          begin
            if displaytype = low
              then
                begin
                  if (x < Xmaxlow) and (y < YmaxLow)
                    then
                      MEM[$A000:(word(y) * XmaxLow + x)] := remap[color] ;
                end
            else if pc_device = pc_VGA
              then
                begin
                  if (x < XmaxHigh) and (y < YmaxHigh)
                    then
                      begin
                        port[$3CE] := 8 ;
                        port[$3CF] := ($80 shr (x and 7)) ;
                        offset := word(y) * 80 + (x shr 3) ;
                        oldval :=  mem[$A000:offset] ;
                        port[$3C4] := 2 ;
                        port[$3C5] := remap[color] ;
                        mem[$A000:offset] := $FF ;
                      end
                  end
              else
                PutPixel(x,y,remap[color]) ;
          end
      else { all other graphical devices }
        PutPixel(x,y,remap[color]) ;

    end ; { Display_Pixel }


  procedure Close_Graph ;

    begin

      CloseGraph ;

    end ;



  procedure CheckDisplay ;

    {
      determines display type,
      returns values for:

        low resolution
        high resolution
        driver software
    }

    var

      code : integer ;
      SVGA : string ;
      BGI_name : PathStr ;

    begin

      { default for IBM PC to 320x200 and 640x? }
      XmaxLow  := 320 ;
      YmaxLow  := 200 ;
      XmaxHigh := 640 ;
      max_colors_low := 256 ;
      max_colors_high := 256 ;
      pc_device := pc_none ;

      DetectGraph(GraphDriver, Graphmode) ;
      case graphdriver of
        CGA      : begin
                     pc_device := pc_CGA ;
                     Graphmodelow  := CGAC2 ;
                     GraphModeHigh := CGAHi ;
                     YmaxHigh := 200 ;
                     BGI_name_low  := 'CGA.BGI' ;
                     BGI_name_high := 'CGA.BGI' ;
                   end ;
        EGA      : begin
                     pc_device := pc_EGA ;
                     Graphmodelow  := CGAC0 ;
                     GraphModeHigh := EGAHi ;
                     YmaxLow := 200 ;
                     BGI_name_low  := 'EGAVGA.BGI' ;
                     BGI_name_high := 'EGAVGA.BGI' ;
                   end ;
        HercMono : begin
                     pc_device := pc_herc ;
                     Graphmodelow  := HercMonoHi ;
                     GraphModeHigh := HercMonoHi ;
                     XmaxLow  := 720 ;
                     YmaxLow  := 348 ;
                     XmaxHigh := 720 ;
                     YmaxHigh := 348 ;
                     BGI_name_low  := 'HERC.BGI' ;
                     BGI_name_high := 'HERC.BGI' ;
                   end ;
        VGA      : begin
                     pc_device := pc_VGA ;
                     Graphmodelow  := 0 ;
                     GraphModeHigh := VGAHi ;
                     YmaxLow := 200 ;
                     BGI_name_low  := 'VGA256.BGI' ;
                     SVGA := GetEnv('SVGA') ;
                     if SVGA = ''
                       then
                         begin
                           YmaxHigh := 480 ;
                           GraphmodeHigh := VGAHi ;
                           BGI_name_high := 'EGAVGA.BGI' ;
                           max_colors_high := 16 ;
                         end
                       else
                         begin
                           pc_device := pc_SVGA ;
                           BGI_name_high := 'SVGA256.BGI' ;
                           max_colors_high := 256 ;
                           { find out which mode supported on this controller }
                           val(SVGA, maxGraphModeHigh, code) ;
                           if (code <> 0) or
                               not (maxGraphmodeHigh in [1,2,3,4])
                             then
                               maxGraphmodeHigh := 1 ;
                           case maxGraphModeHigh of
                             1 : YmaxHigh := 400 ;
                             2 : YmaxHigh := 480 ;
                             3 : begin
                                   XmaxHigh := 800 ;
                                   YmaxHigh := 600 ;
                                 end ;
                             4 : begin
                                   XmaxHigh := 1024 ;
                                   YmaxHigh := 768  ;
                                 end ;
                           end ;
                         end ;
                   end ;
      end ;

      if pc_device = pc_none
        then
          writeln('No supported garphical device found')
        else
          begin

            write('Graphical device found: ') ;
            case pc_device of
              pc_CGA  : write('CGA in 320x200 and 640x200') ;
              pc_EGA  : write('EGA in 320x200 and 640x200') ;
              pc_HERC : write('Hercules 720x348') ;
              pc_VGA  : write('VGA in 320x200 256 colors and 640x480 16 colors') ;
              pc_SVGA : begin
                          write('SVGA in 320x200 256 colors and ') ;
                          case maxGraphModeHigh of
                            1 : write('640x400') ;
                            2 : write('640x480') ;
                            3 : write('800x600') ;
                            4 : write('1024x768') ;
                          end ;
                          write(' in 256 colors') ;
                        end ;
            end ;
            writeln ;

            { try to locate ?.BGI on the path }
            S_low := FSearch(BGI_name_low,GetEnv('PATH'));
            if S_low = ''
              then
                begin
                  WriteLn('File ',
                          BGI_name_low,
                          ' not found on PATH ') ;
                  pc_device := pc_none
                end
              else
                { split filespec into directory path info for InitGraph }
                Fsplit(FExpand(S_low),Dir_low,Name_low,Ext_low) ;

            S_high := FSearch(BGI_name_high,GetEnv('PATH'));
            if S_high = ''
              then
                begin
                  WriteLn('Bestand ',
                          BGI_name_high,
                          ' not found on PATH') ;
                  pc_device := pc_none ;
                end
              else
                { split filespec into directory path info for InitGraph }
                Fsplit(FExpand(S_high),Dir_high,Name_high,Ext_high) ;
          end ;


  end ; { CheckDisplay }

end. { unit SGRAPH }