unit ssixel ;

{

   Module  : Process sixel file

   Author  : Hans Otten

   Version : 1.3  3-feb-1991

   Facility: MSX/PCX/Sixel routines

   Purpose : Process tokenized sixel file

  }

interface

  uses

    CNV  ;


  procedure scan_sixel (var parse_info : parse_infotype ;
                        var colormap   : colormap_type ;
                        display_info   : display_infotype) ;

  procedure display_sixel(parse_info   : parse_infotype ;
                          colormap     : colormap_type ;
                          display_info : display_infotype) ;



implementation

  uses

    DOS, CRT,
    SBUFFER, SGRAPH, SCREATE ;

  const

    ST  = #156 ;
    ESC = #27 ;
    DCS = #144 ;

  type

    token_kind = (
                  string_introducer,    { DCS 90 or ESC P                 }
                  string_terminator,    { ST 9C of ESC \                  }
                  separator,            { ;                               }
                  parameter,            { decimal value,                  }
                  sixel_char,           { 3F to 7E, ? to ~                }
                  graphics_repeat,      { !                               }
                  raster_attributes,    { "                               }
                  color_introducer,     { #                               }
                  graphics_CR,          { $                               }
                  graphics_newline,     { -                               }
                  other_char,           { not in sixel protocol, ignored  }
                  endoffile             { not in sixel protocol,
                                          unexpected end of file          }
                 ) ;

      tokentype = record
                    kind  : token_kind ;
                    value : integer ;
                  end ;


  var

    error_count : integer ;    { keep track of nr of errors    }
    previous_token,            { for pushback                  }
    token : tokentype ;        { returned by get_token         }
    next_char  : char;         { next character from input     }
    max_row,                   { max sixel row (multiple of 6  }
    max_col,                   { maximum found column position }
    sixel_row,                 { current sixel line (mult of 6 }
    sixel_col : integer ;      { Current sixel column position }
    first,                     { true if first newline skipped }
    pushback : boolean ;       { true if token pushed back     }

    sixel_buffer : array[1..max_pixel_col, 1..6] of byte ;


  procedure get_token;

    {
      Return next token from the sixel input file

      The routine returns information in
       -  record token with two fields

           token.kind : The kind of token
           token.value :
             Sixel character: The 6 bit sixel value
             Decimal number : The number
             unknown char   : ord of char

       next_char : The last character read


       The token parsing is ordered in such a way that the
       tokens with the highest frequency are parsed first

    }



    begin { main get_token }

      previous_token := token ;
      if pushback
        then
          pushback := false
        else
          with token do
            begin
              value := 0 ;
              if (next_char >= '?') and (next_char <= '~')
                then
                  begin
                    kind  := sixel_char ;
                    value :=ord(next_char)-ord('?') ;
                    GetNextChar(next_char) ;
                  end
              else if next_char = '!'
                then
                 begin
                   kind := graphics_repeat ;
                   GetNextChar(next_char) ;
                end
             else if (next_char >= '0') and (next_char <= '9')
               then
                 begin
                   value := 0 ;
                   kind := parameter ;
                   { add to parameter while decimal  }
                   repeat
                     value := (10 * value) + ord(next_char) - ord('0') ;
                     GetNextChar(next_char)
                   until (next_char < '0') or (next_char > '9') ;
                 end
              else
                begin
                  if next_char = ';'
                    then
                      kind := separator
                  else if next_char = '#'
                    then
                      kind := color_introducer
                  else if next_char = '$'
                    then
                      kind := graphics_CR
                  else if next_char = '-'
                    then
                      kind := graphics_newline
                  else if next_char = ESC
                    then
                      begin
                        GetNextChar(next_char) ;
                        if next_char in ['P','\']
                          then
                            begin
                              case next_char of
                                'P' : kind := string_introducer ;
                                '\' : kind := string_terminator ;
                                end ;
                        end
                  end
              else if next_char = ST
                then
                  kind := string_terminator
              else if next_char = DCS
                then
                  kind := string_introducer
              else if next_char = '"'
                then
                  kind := raster_attributes
              else if next_char = CTRLZ
                then
                  kind := endoffile
              else
                begin
                  kind  := other_char ;
                  value := ord(next_char)  ;
                end ;
              GetNextChar(next_char) ;
            end ;
        end ;

  end ; { get_token }


  procedure init_token ;

    begin

      GetNextChar(next_char) ;
      pushback := false ;
      token.value := 0 ;
      token.kind := endoffile ;

   end ;


  procedure parse_DCS (var parse_info : parse_infotype ;
                       display_info : display_infotype) ;


    var

      count : integer ;

    begin

      init_token ;
      { skip all in front of DCS }
      repeat
        repeat
          get_token ;
          if token.kind <> string_introducer
            then
              parse_info.DCS_skipped := true ;
        until ( ( token.kind = endoffile         ) or
                ( token.kind = string_introducer ) ) ;
        if token.kind = endoffile
          then
            report_error(premature_end,fatal, display_info.display) ;
        get_token ;
      until (token.kind = separator                   ) or
            ( (token.kind  = sixel_char          ) and
              (token.value = ord('q') - ord('?') )    ) or
            (token.kind = parameter                   ) ;
      pushback := true ;

      if parse_info.DCS_skipped
        then
          report_error(skipped_dcs,warning,display_info.display) ;

      parse_info.pcount := 0 ;
      repeat
        get_token ;
        if (token.kind = parameter)
          then
            begin
              parse_info.pcount := parse_info.pcount + 1 ;
              parse_info.ps[parse_info.pcount] := token.value ;
             end
        else if not (
                     (  token.kind = separator             ) or
                      ( (token.kind  = sixel_char       ) and
                        (token.value = ord('q') - ord('?')  )  )  )
          then
            report_error(invalid_dcs, fatal,display_info.display) ;

      until  ( (token.kind  = sixel_char          ) and
             (token.value = ord('q') - ord('?') )    ) or
             (  token.kind  = endoffile               ) ;
      if token.kind = endoffile
            then
              report_error(premature_end,fatal,display_info.display) ;

    end ; { parse_DCS }

  procedure parse_raster_attributes(var parse_info : parse_infotype;
                                    display_info : display_infotype) ;

    {
      checks raster attributes,
      must follow DCS immediately
    }

    var

      count : integer ;

    begin

      { check raster attributes }

      parse_info.raster_valid := false ;
      get_token ;
      if token.kind = raster_attributes
        then
          begin
            parse_info.raster_valid := true ;
            { get four parameters }
            count := 1 ;
            while count <= 4 do
              begin
                get_token ;
                if token.kind <> parameter
                  then
                    begin
                      pushback := true ;
                      count := 5 ;
                    end
                  else
                    parse_info.r_attributes[count] := token.value ;
                get_token ;
                if token.kind <> separator
                  then
                    begin
                      pushback := true ;
                      count := 5 ;
                    end
                  else
                    inc(count) ;
              end ;
          end
        else
          pushback := true ;

    end ; { raster_attributes }

  procedure parse_picture_data (var parse_info : parse_infotype ;
                                var colormap : colormap_type ;
                                display_info : display_infotype) ;

    var

      col_count,
      row_count,
      color_count,
      count         : integer ;
      ch : char ;

    procedure Flush_sixel_buffer ;

      {
        send all pixels in buffer to output,
        offset in picture row is sixel_row
        clears sixelbuffer with colornr 0
      }

      var

        col_count,
        row_count  : integer ;

      begin

        {  flush sixel_buffer to Add_Pixel  }

        for row_count := 1 to 6 do
          for col_count := 1 to parse_info.max_parse_col do
            begin
              Add_pixel(col_count,
                        row_count + sixel_row - 1,
                        sixel_buffer[col_count,row_count]) ;
              sixel_buffer[col_count,row_count] := 0 ;
            end ;

      end ; { Flush_sixel_buffer }

    procedure parse_color ;

      {
        parse color sequense,
        can be color selection for next write
        or color specification
        # already read, next must be color nr
      }

      var

        rgb      : boolean ;
        color_count : color_values ;
        count,
        color_nr : integer ;

      procedure convert_hls_to_rgb (var hr, lg, sb : byte) ;

        {
          convert h, l and s to r, g and b values 0 .. 255
          based upon Foley and Van Dam,
        }

        var

          m1, m2 ,
          r, g, b,
          h, l, s  : real ;

        function value (n1, n2, hue: real) : real ;

          begin

            if h > 360 then h := h - 360 ;
            if h < 0   then h := h + 360 ;
            if hue < 60.0
              then
                value := n1 + (n2 - n1) * (hue / 60.0)
            else if hue < 180.0
              then
                value := n2
            else if hue < 240.0
              then
                value := n1 + (n2 - n1) * (240.0 - hue) / 60.0
            else
              value := n1

          end ; { value }

        function convert_to_int ( arg : real ) : integer ;

          { converts real arg in [0,1] to integer in [0,255] }

          var

            i : integer ;

          begin

            i := round(arg * colormax) ;
            if i < 0
              then
                i := 0
            else if i > colormax
              then
                i := colormax ;
            convert_to_int := i ;

          end ;

        begin

          { convert to real, l and s to range [0,1] }
          h := hr - 120 ;  { compensate for difference Tektronix
                             and Foley + van Dam model           }
          if h < 0
            then
              h := h + 360.0 ;

          l := lg / 100.0 ;
          s := sb / 100.0 ;

          if l <= 0.5
            then
              m2 := l + (l * s)
            else
              m2 := l + s - (l * s) ;
          m1 := 2.0 * l - m2 ;
          if s = 0
            then
              { achromatic case: make grey scale }
              begin
                r := l ;
                g := l ;
                b := l ;
              end
            else
              begin
                r := value(m1, m2, h + 120.0) ;
                g := value(m1, m2, h) ;
                b := value(m1, m2, h - 120.0) ;
              end ;

          hr := convert_to_int(r) ;
          lg := convert_to_int(g) ;
          sb := convert_to_int(b) ;

        end ; { convert_hls_to_rgb }


      begin { main parse_color }

        get_token ;
        if token.kind = parameter
          then
            color_nr := token.value
          else
            report_error(invalid_color,fatal,display_info.display) ;

        get_token ;
        if token.kind <> separator
          then
            begin
              { we have color selection, give token back }
              pushback := true ;
              if color_nr <= max_color
                then
                  begin
                    if display_info.display = parse
                      then
                        colormap[color_nr].used := true
                      else
                        parse_info.current_color := color_nr ;
                  end
                else
                  report_error(invalid_color,fatal,display_info.display) ;
            end
          else
            begin
              get_token ;
              if token.kind <> parameter
                then
                  report_error(invalid_color,fatal,display_info.display)
                else
                  { color specification }
                  begin
                    if display_info.display = parse
                      then
                        colormap[color_nr].defined := true ;
                    if token.value = 1
                      then
                        rgb := false
                    else if token.value = 2
                      then
                        rgb := true
                    else
                      report_error(invalid_color,fatal,display_info.display) ;
                    for color_count := c_red to c_blue do
                      begin
                        get_token ;
                        if token.kind <> separator
                          then
                            report_error(invalid_color,
                                         fatal,display_info.display) ;
                        get_token ;
                        if token.kind <> parameter
                          then
                            report_error(invalid_color,
                                         fatal,display_info.display)
                          else
                            if display_info.display = parse
                              then
                                 colormap[color_nr].colors[color_count]
                                   := round(token.value * colormax / 100) ; ;
                      end ;
                    { second and third must be 0 to 255 }
                    if (colormap[color_nr].colors[c_blue] > colormax) or
                       (colormap[color_nr].colors[c_green] > colormax)
                      then
                        report_error(invalid_color,fatal,display_info.display) ;
                    if rgb
                      then
                        begin
                          if colormap[color_nr].colors[c_red] > colormax
                            then
                              report_error(invalid_color,fatal,display_info.display)
                        end
                      else
                        begin
                          if colormap[color_nr].colors[c_red] > 360
                            then
                              report_error(invalid_color,fatal,display_info.display)
                          else
                            if display_info.display = parse
                              then 
                                convert_hls_to_rgb
                                 (colormap[color_nr].colors[c_red],
                                  colormap[color_nr].colors[c_green],
                                  colormap[color_nr].colors[c_blue]) ;
                        end ;
                  end ;
            end ;

      end ; { parse_color }


    procedure add_sixel(sixel_value : integer) ;

      { adds current sixel value to output }

      var

        count : integer ;


      begin { main add_sixel }

        if display_info.display <> parse
          then
            begin
              for count := 1 to 6 do
                begin
                  if odd(sixel_value)
                    then
                      sixel_buffer[sixel_col, count] :=
                       parse_info.current_color ;
                  sixel_value := sixel_value shr 1 ;
                end ;
            end ;
        sixel_col := sixel_col + 1 ;

      end ; { add_sixel }

    procedure start_repeat ;

      var

        count,
        count_repeat : integer ;

      begin

        get_token ;
        if token.kind <> parameter
          then
            report_error(invalid_repeat,fatal,display_info.display)
          else
            count_repeat := token.value ;
        get_token ;
        if token.kind <> sixel_char
          then
            report_error(invalid_repeat,fatal,display_info.display)
          else
            for count := 1 to count_repeat do
              add_sixel(token.value) ;

      end ; { start_repeat }


    procedure process_graphics_CR ;

      begin

        if sixel_col > max_col
          then
            max_col := sixel_col ;
        sixel_col := 1 ;

      end ; { process_graphics_CR }


    procedure process_newline ;

      var

        ch : char ;


      begin
        if keypressed
          then
            begin
              if display_info.display = video
                then
                  Close_Graph ;
              token.kind := string_terminator ;
              pushback := true ;
            end
        else
          begin
            { if we start with newline : ignore }
            if not ( (sixel_col = 1) and
                     (sixel_row = 1) and
                     (max_col = 0)   and
                     first                )
               then
                 begin
                   if display_info.display <> parse
                     then
                       Flush_sixel_buffer ;
                   sixel_row := sixel_row + 6 ;
                   process_Graphics_CR ;
                   if display_info.display <> video
                     then
                       write(#08,#08,#08,sixel_row:3) ;
                   if display_info.display <> parse
                     then
                       begin
                         { force stop of display if outside window }
                         if sixel_row >= display_info.end_view_row
                           then
                             begin
                               token.kind := string_terminator ;
                               pushback := true ;
                             end ;
                       end
                 end
               else
                 first := true ;
          end ;
      end ; { process_newline }

    begin { main parse_picture_data }

      first := false ;
      sixel_row := 1 ;
      sixel_col := 1 ;
      max_col := 0 ;
      { clear sixel buffer }
      if display_info.display <> parse
        then
          for col_count := 1 to parse_info.max_parse_col do
            for row_count := 1 to 6 do
              sixel_buffer[col_count, row_count] := 0 ;

      if display_info.display <> video
        then
          write('Working on sixel line    ') ;
      repeat
        get_token ;
        case token.kind of
          sixel_char       : add_sixel(token.value) ;
          graphics_repeat  : start_repeat ;
          graphics_CR      : process_graphics_CR ;
          graphics_newline : process_newline ;
          color_introducer : parse_color ;
        end ;
      until ( ( token.kind = endoffile         ) or
              ( token.kind = string_terminator ) ) ;
      if token.kind = endoffile
        then
          report_error(premature_end, fatal,display_info.display) ;
      { if we do not end with newline force it }
      if previous_token.kind <> graphics_newline
        then
          begin
            process_newline ;
            { correct for one sixel row too far }
            sixel_row := sixel_row - 6 ;
          end ;
      if display_info.display = video
        then
          begin
             writeln(chr(7)) ;
             repeat until keypressed ;
          end
        else
          begin
            writeln ;
            writeln('Converted') ;
          end ;
      if display_info.display = parse
        then
          begin
            if (parse_info.r_attributes[4] < sixel_row) and
               (abs(parse_info.r_attributes[4] - sixel_row) < 6)
              then
                parse_info.max_parse_row := parse_info.r_attributes[4]
              else
                parse_info.max_parse_row := sixel_row ;
            parse_info.max_parse_col := max_col - 1 ;
            if parse_info.max_parse_col > max_pixel_col
              then
                report_error(toobig,fatal,display_info.display) ;
         end ;

    end ; { parse_picture_data }

  procedure scan_sixel (var parse_info : parse_infotype ;
                        var colormap   : colormap_type ;
                        display_info   : display_infotype) ;

    begin

      parse_DCS(parse_info, display_info) ;
      parse_raster_attributes(parse_info,display_info) ;
      parse_picture_data(parse_info, colormap, display_info) ;

    end ; { scan_sixel }

  procedure display_sixel(parse_info   : parse_infotype ;
                          colormap     : colormap_type ;
                          display_info : display_infotype) ;
    begin

      parse_DCS(parse_info, display_info) ;
      parse_raster_attributes(parse_info, display_info) ;
      parse_picture_data(parse_info, colormap, display_info) ;

    end ; { display_sixel }

end. { unit stoken }
