unit smsxcr ;

 interface

 {
   Module  : SMSXCR

   Author  : Hans Otten

   Version : 2.0  16-sep-1991

   Facility: Sixel/PCX/MSX routines

   Purpose : output pixels to
               - MSX Copy or Screen, 5-8, STP, screen 12


  }
 uses

   CNV  ;

  procedure Init_output_MSX(display_info : display_infotype ;
                            parse_info : parse_infotype ;
                            colormap : colormap_type) ;

  procedure Close_output_MSX ;

  procedure Add_pixel_MSX(x,y,color : integer );


implementation

  uses

    DOS, CRT,
    SOUTBUF ;

  const

                        { R    G    B   }
     remap57 : array[0..7,0..7,0..7] of byte =

{--------------------------------------------------------------------
  R    B=0  1  2  3  4  5  6  7     0  1  2  3  4  5  6  7  B    G
 --------------------------------------------------------------------}
{ 0 }((( 1, 1, 4, 4, 4, 4, 4, 4), ( 1, 1, 4, 4, 4, 4, 4, 4),   { 0-1  }
{ 0 }  ( 1, 1, 1, 4, 4, 4, 4, 4), (12,12,12, 4, 4, 4, 4, 4),   { 2-3  }
{ 0 }  (12,12,12,12, 4, 4, 5, 5), (12,12, 2, 2, 5, 5, 5, 5),   { 4-5  }
{ 0 }  ( 2, 2, 2, 2, 3, 3, 7, 7), ( 2, 2, 2, 3, 3, 5, 7, 7)),  { 6-7  }
{ 1 } (( 1, 1, 1, 4, 4, 4, 4, 4), ( 1, 1, 4, 4, 4, 4, 4, 4),   { 0-1  }
{ 1 }  ( 1,12,12, 4, 4, 4, 4, 4), (12,12,12,12, 4, 4, 5, 5),   { 2-3  }
{ 1 }  (12,12,12,12, 4, 5, 5, 5), ( 2, 2, 2, 2, 5, 5, 5, 5),   { 4-5  }
{ 1 }  ( 7, 7, 7, 7, 7, 7, 7, 7), ( 3, 3, 3, 3, 3, 7, 7, 7)),  { 6-7  }
{--------------------------------------------------------------------
  R    B=0  1  2  3  4  5  6  7     0  1  2  3  4  5  6  7  B    G
 --------------------------------------------------------------------}
{ 2 } (( 1, 6, 6, 6, 4, 4, 4, 4), ( 1, 6, 6, 6, 4, 4, 4, 4),   { 0-1  }
{ 2 }  ( 1, 1, 1, 6, 4, 4, 4, 4), (12,12,12, 4, 4, 4, 5, 5),   { 2-3  }
{ 2 }  (12,12,12,12, 5, 5, 5, 5), (12,12, 2, 2, 2, 5, 7, 7),   { 4-5  }
{ 2 }  ( 2, 2, 2, 3, 7, 7, 7, 7), ( 3, 3, 3, 3, 3, 3, 7, 7)),  { 6-7  }
{ 3 } (( 6, 6, 6, 6, 6, 4, 4, 4), ( 6, 6, 6, 6, 6, 4, 4, 7),   { 0-1  }
{ 3 }  ( 6, 6, 6, 6, 6, 4, 4, 4), ( 4,12,12,14, 4, 5, 5, 5),   { 2-3  }
{ 3 }  (12,12,12,12, 4, 5, 5, 5), (12,12,12,12, 5, 5, 7, 7),   { 4-5  }
{ 3 }  ( 2, 2, 2, 3, 3, 7, 7, 7), ( 3, 3, 3, 3, 3, 7, 7, 7)),  { 6-7  }
{--------------------------------------------------------------------
  R    B=0  1  2  3  4  5  6  7     0  1  2  3  4  5  6  7  B    G
 --------------------------------------------------------------------}
{ 4 } (( 6, 6, 6,13,13,13, 4, 4), ( 6, 6, 6,13,13, 4, 4, 4),   { 0-1  }
{ 4 }  ( 6, 6, 6,13,13,13, 4, 4), ( 6, 6, 6,13,13, 5, 5, 5),   { 2-3  }
{ 4 }  (12,12, 2, 2,14,14, 5, 5), ( 2, 2, 2, 2, 3, 7, 7, 7),   { 4-5  }
{ 4 }  ( 2, 2, 2, 3, 3, 7, 7, 7), ( 3, 3, 3, 3, 3, 3, 7, 7)),  { 6-7  }
{ 5 } (( 6, 6, 6, 6,13,13,13,13), ( 6, 6, 6, 6,13,13,13, 4),   { 0-1  }
{ 5 }  ( 6, 6, 6,13,13,13,13,13), ( 8, 8, 8, 8,13,13,13, 4),   { 2-3  }
{ 5 }  (10,10,10,13,13,13, 4, 4), (10,10,10,10,14,14,14, 7),   { 4-5  }
{ 5 }  ( 3, 3, 3,10,10,14,14, 7), ( 3, 3, 3, 3, 3, 7, 7, 7)),  { 6-7  }
{--------------------------------------------------------------------
  R    B=0  1  2  3  4  5  6  7     0  1  2  3  4  5  6  7  B    G
 --------------------------------------------------------------------}
{ 6 } (( 8, 8, 8,13,13,13,13,13), ( 8, 8, 9, 9,13,13,13,13),   { 0-1  }
{ 6 }  ( 8, 8, 8,13,13,13,13,13), ( 9, 9, 9,13,13,13,13,13),   { 2-3  }
{ 6 }  ( 9, 9, 9, 9,13,13,13,13), (10,10,10,10,10, 9,13,13),   { 4-5  }
{ 6 }  (10,10,11,11,11,11,14,14), (10,10,10,11,11,11,11,14)),  { 6-7  }
{ 7 } (( 8, 8, 8, 8,13,13,13,13), ( 8, 8, 8, 8, 8,13,13,13),   { 0-1  }
{ 7 }  ( 8, 8, 8, 9, 9,13,13,13), ( 9, 9, 9, 9, 9,13,13,13),   { 2-3  }
{ 7 }  (10,10, 9, 9,13,13,13, 0), (10,10,10,10, 9,13,13,13),   { 4-5  }
{ 7 }  (10,10,10,10,11,11,15, 0), (10,10,10,11,11,15,15,15))); { 6-7  }
{--------------------------------------------------------------------
  R    B=0  1  2  3  4  5  6  7     0  1  2  3  4  5  6  7  B    G
 --------------------------------------------------------------------}


  var

    palettefile    : text ;
    output_buffer : array[1..8192] of byte ;
    previous_row,
    dumped_lines,                         { nr of pixel lines written }
    max_line_inuse : integer ;
    remap : array [0..max_color] of byte ;
    msxcolormap : colormap_type ;
    cdisplay_info : display_infotype ;

    { variables for MSX copy output }
    inpixel_count,               { 1 if byte_value empty }
    byte_value,
    msx_bytes,
    pixels_to_write,
    pixels_written : integer ;   { count of pixels written sofar
                                   to scanline }

    scr12c_RGB : array[0..3, c_red..c_blue] of byte ;

  procedure Init_output_MSX(display_info : display_infotype ;
                            parse_info : parse_infotype ;
                            colormap : colormap_type) ;

    var

      x,y,
      count : integer ;
      palettename : string ;
      no_palette  : boolean ;


      procedure fill_screen57_remap ;

        { screen 5 and have 16 colors available
          if used_colors <= 16
            then we can generate a palette
          else
            map to default screen colors
        }

        var

          r,g,b : byte ;
          color_count,
          remap_count : integer ;

        procedure get_palette_name ;

          var

            ch : char ;
            default_name : string ;

          begin

            with display_info do
              begin
                default_name := '.PL' ;
                if msx_screen = scr5
                  then
                    default_name := 'pl5'
                else if msx_screen = scr6
                  then
                    default_name := 'pl6'
                else if msx_screen = scr7
                  then
                    default_name := 'pl7'
              end ;

            default_name := parse_info.filename + '.' + default_name ;
            repeat
              write('Naam palette file (ESCAPE voor geen) [',
                     default_name,']? ') ;
              Get_String(palettename) ;
              no_palette := (palettename = chr(27)) ;
              writeln ;
              if length(palettename) = 0
                then
                  palettename := default_name ;
              assign(palettefile,palettename) ;
              {$I-}
              rewrite(palettefile) ;
              {$I+}
            until (IOResult = 0) or no_palette ;

          end ; { get_palette_name }

        begin

          palettename := '' ;
          if (parse_info.used_colors <= 16) and
             (display_info.msx_screen <> scr8) and
             (display_info.msx_screen <> scr12)
            then
              get_palette_name ;
          if not no_palette
            then
              begin
                { write BLOAD header }
                write(palettefile,chr($FE)) ;
                if display_info.msx_screen = scr5
                  then
                    { palette table $7680 $769F }
                    write(palettefile,chr($80),chr($76), chr($9F),chr($76))
                  else
                    { palette table $FA80 $FA9F }
                    write(palettefile,chr($80),chr($FA), chr($9F),chr($FA)) ;
                { dummy start address }
                write(palettefile,chr(0),chr(0)) ;

                remap_count := 0 ;
                for color_count := 0 to max_color do
                  begin
                    if colormap[color_count].used
                      then
                        begin
                          r :=
                            round(colormap[color_count].colors[c_red]
                                       * 7/colormax) ;
                          g :=
                            round(colormap[color_count].colors[c_green]
                                       * 7/colormax) ;
                          b :=
                            round(colormap[color_count].colors[c_blue]
                                       * 7/colormax) ;
                          remap[color_count] := remap_count ;
                          inc(remap_count) ;
                          r := r shl 4 ;
                          write(palettefile,chr(r+b),chr(g)) ;
                        end
                  end ;
                for color_count := 1 to (16 - parse_info.used_colors) do
                  write(palettefile,chr(0),chr(0) ) ;
                close(palettefile)
              end   ;
          if no_palette { palettefile problem or none wanted }
            then
              { try to match to standard palette max 15 colors ! }
              begin
                for color_count := 0 to max_color do
                  if colormap[color_count].used
                    then
                      remap[color_count] :=
                      remap57[round(colormap[color_count].colors[c_red]*7/colormax),
                              round(colormap[color_count].colors[c_green]*7/colormax),
                              round(colormap[color_count].colors[c_blue]*7/colormax)]
              end ;


        end ; { fill_screen57_remap }


      procedure fill_screen6_remap ;

        {
          screen 6 has 4 colors available
                          color 0  =   0  0  0
                          color 1  =   0  0  0
                          color 2  =   1  6  1
                          color 3  =   3  6  1  }
        var

          color_nr : integer ;

        begin

          for color_nr := 0 to max_color do
            begin
              if colormap[color_nr].colors[c_green] > 182
                then  { dark or light green }
                  begin
                    if colormap[color_nr].colors[c_red] > 73
                      then
                        remap[color_nr] := 3
                      else
                       remap[color_nr] :=  2
                  end
                else  { black }
                  remap[color_nr] := 1
            end ;


        end ; { fill_screen6_remap }

      procedure fill_stp6_remap ;

          { stempels have color 0  =   7  7  7
                          color 1  =   0  0  0
                          color 2  =   0  0  0
                          color 3  =   0  0  0
            colors are weighted and digitized to black and white
          }


        var

          color_nr : integer ;
          level    : integer ;

        begin

          level := 80 ;
          if not parse_info.mono
            then
              repeat
                write('Digitize level (0 - 255) [',level:1,']? ') ;
                level := GetValue(level) ;
              until (level > 0) and (level < 256) ;
          for color_nr := 0 to max_color do
            begin
              if ( colormap[color_nr].colors[c_red] +
                   colormap[color_nr].colors[c_green] +
                   colormap[color_nr].colors[c_blue] ) > (level * 3)
                then
                  remap[color_nr] := 0
                else
                  remap[color_nr] := 1
            end ;
        end ;


      procedure fill_screen8_remap ;

        {
          screen 8 colormap is fixed:
           R = 0 .. 7  ( 0 36 73 109 146 182 219 255)
           G = 0 .. 7  ( 0 36 73 109 146 182 219 255)
           B = 0 .. 3  ( 0 85 170 255)

           to map colormap to screen 8 the RGB values are checked to
           be within range of the fixed values, border is midpoint range
           for RG the range is fixed value +/- 6.125
           for B  the range is fixel value +/- 16.6
           rounded to integer values:
           RG 0 = < 18   1 = < 55  2 = < 91 3 = < 128
              4 = < 164  5 = < 200 6 = < 237
           B  0 = < 43   1 = < 128 2 = < 213
        }

        var

          r, g, b : integer ;
          color_count : integer ;

        begin
          for color_count := 0 to max_color do
            begin
              if colormap[color_count].used
                then
                  { calculate screen 8 byte from colormap values }
                  begin
                    r := colormap[color_count].colors[c_red];
                    g := colormap[color_count].colors[c_green] ;
                    b := colormap[color_count].colors[c_blue] ;
                    if r < 18
                      then
                        r := 0
                    else if r < 55
                      then
                        r := 1
                    else if r < 91
                      then
                        r := 2
                    else if r < 128
                      then
                        r := 3
                    else if r < 164
                      then
                        r := 4
                    else if r < 200
                      then
                        r := 5
                    else if r < 237
                      then
                        r := 6
                    else
                      r := 7 ;


                    if g < 18
                      then
                        g := 0
                    else if g < 55
                      then
                        g := 1
                    else if g < 91
                      then
                        g := 2
                    else if g < 128
                      then
                        g := 3
                    else if g < 164
                      then
                        g := 4
                    else if g < 200
                      then
                        g := 5
                    else if g < 237
                      then
                        g := 6
                    else
                      g := 7 ;

{
           B  0 = < 43   1 = < 128 2 = < 213
}
                    if b < 43
                      then b := 0
                    else if b < 128
                      then
                        b := 1
                    else if b < 213
                      then
                        b := 2
                    else
                      b := 3 ;

                    { shift r and g bits to correct position }
                    r := r shl 2 ;
                    g := g shl 5 ;
                    remap[color_count] := r + g + b ;
                  end ;
            end ;

          end ; { fill_screen8_remap }

    procedure generate_loader ;

      { produce simple basic program to display msx file }

      var

        msx_loadername : string ;
        screen : integer ;

      begin

        with display_info do
          begin
            msx_loadername := '.LDR' ;
            if msx_type = mscreen
              then
                msx_loadername := '.LS'
              else
                msx_loadername := '.LC' ;
            case msx_screen of
              scr5 : begin
                       screen := 5 ;
                       msx_loadername := msx_loadername + '5' ;
                     end ;
              scr6 : begin
                       screen := 6 ;
                       msx_loadername := msx_loadername + '6' ;
                     end ;
              scr7 : begin
                       screen := 7 ;
                       msx_loadername := msx_loadername + '7' ;
                     end ;
              scr8 : begin
                       screen := 8 ;
                       msx_loadername := msx_loadername + '8' ;
                     end ;
              scr12 : begin
                       screen := 12 ;
                       msx_loadername := msx_loadername + 'C' ;
                     end ;
            end ;

            msx_loadername := parse_info.filename + msx_loadername ;

            assign(palettefile, msx_loadername) ;
            {$I-}
            rewrite(palettefile) ;
            {$I+}
            if IOresult <> 0
              then
                exit ;
            writeln(palettefile, '1 REM CNVS loader file (C) Hans Otten') ;
            writeln(palettefile, '10 SCREEN ',screen:1) ;
            if not no_palette
              then
                writeln(palettefile, '20 VDP(9) = VDP(9) OR 32') ;

            writeln(palettefile, '30 COLOR ,0,0 : CLS') ;
            if not no_palette
              then
                begin
                  writeln(palettefile, '40 BLOAD "',palettename,'",s') ;
                  writeln(palettefile, '50 COLOR = RESTORE') ;
               end
            else if msx_screen = stp6
              then
                writeln(palettefile,
                          '40 COLOR 0=(7,7,7) : COLOR 1 = (0,0,0)') ;

            write(palettefile, '60 ') ;

            if msx_type = mscreen
              then
                writeln(palettefile,'BLOAD "',
                                     display_info.filename,
                                     display_info.filext,'",S')
              else { copy }
                writeln(palettefile,'COPY "',
                                     display_info.filename,
                                     display_info.filext,'" TO (0,0)') ;
            writeln(palettefile, '70 IF INKEY$ = "" THEN 70') ;
            writeln(palettefile, '80 SCREEN 0 : END') ;

            close(palettefile) ;
          end ;

      end ;

    begin { main init_output_msx }

      cdisplay_info := display_info ;
      msxcolormap := colormap ;
      no_palette := true ;
      with cdisplay_info do
        begin
          { calculate nr of bytes in MSX screen line }
          { screen 8 and 12 1 pixel per byte }
          pixels_to_write := (end_view_col - start_view_col + 1) ;
          { screen 5 and 7 two pixel per byte }
          if (msx_screen = scr5) or (msx_screen = scr7)
            then
              begin
                msx_bytes := pixels_to_write div 2 ;
                if odd(pixels_to_write)
                  then
                    inc(msx_bytes) ;
              end
          { screen 6 four pixels per byte }
          else if (msx_screen = scr6)
            then
              begin
                msx_bytes := pixels_to_write div 4 ;
                if (pixels_to_write mod 4) > 0
                  then
                    inc(msx_bytes) ;
              end
          else  { screen 8 and 12 }
            msx_bytes := pixels_to_write ;

          for count := 1 to msx_bytes do
            output_buffer[count] := 0 ;

          previous_row := 1 ;
          dumped_lines := 0 ;
          { write screen header }
          if msx_type = mscreen
            then
              begin
                { write BSAVE-ID byte }
                pixel_write($FE) ;
                { write begin address }
                pixel_write(0) ;
                pixel_write(0) ;
                { write end-address }
                if (msx_screen = scr7) or
                   (msx_screen = scr8) or
                   (msx_screen = scr12)
                  then
                    begin
                      pixel_write($FF) ;
                      pixel_write($D3) ;
                    end
                  else { screen 5, 6 }
                    begin
                      pixel_write($FF) ;
                      pixel_write($69) ;
                    end ;
                { write start address }
                pixel_write(0) ;
                pixel_write(0) ;
              end
          else if msx_type = mcopy
            then
              { Copy, so only size to enter }
              begin
                x := end_view_col - start_view_col + 1 ;
                y := end_view_row - start_view_row + 1 ;
                pixel_write(x mod 256) ;
                pixel_write(x div 256) ;
                pixel_write(y mod 256) ;
                pixel_write(y div 256) ;
                { init variables for output }
                pixels_written := 0 ;
                inpixel_count := 1 ;
              end ;

          { fill remap array of colors }
          if msx_screen = scr8
            then
              fill_screen8_remap
          else if (msx_screen = scr5) or (msx_screen = scr7)
            then
              fill_screen57_remap
          else if msx_screen = scr6
            then
              fill_screen6_remap
          else if msx_screen = stp6
            then
              fill_stp6_remap
          else if msx_screen = scr12
            then
              for count := 0 to max_color do
                remap[count] := count ;

        end ;

      generate_loader ;

    end ; { Init_Output_MSX }


  procedure add_msx_copy(color : byte) ;

      { add pixel value to output buffer, MSX screen dependent }

      begin

        case cdisplay_info.msx_screen of
          scr5, scr7 :  begin
                          if inpixel_count = 1
                            then
                              byte_value := color shl 4
                            else
                              begin
                                pixel_write(byte_value or color) ;
                                inpixel_count := 0 ;
                              end ;
                        end ;
          scr6, stp6 :  begin
                          if color <> 0
                            then
                              color := color ;
                          case inpixel_count of
                            1 : byte_value := color shl 6 ;
                            2 : byte_value := byte_value or (color shl 4) ;
                            3 : byte_value := byte_value or (color shl 2) ;
                            4 : begin
                                  pixel_write(byte_value or color) ;
                                  inpixel_count := 0 ;
                                end ;
                          end ;
                      end ;
          scr8       : pixel_write(color) ;


        end ;

      inc(inpixel_count) ;

    end ; { add_msx_copy }


  procedure Flush_Buffer ;

    { for MSX screen flush output buffer contents to device }

    var

      line,
      count : integer ;

    begin

      with cdisplay_info do
        begin
          inc(dumped_lines) ;
          for count := 1 to msx_bytes do
            begin
              pixel_write(output_buffer[count]) ;
              output_buffer[count] := 0 ;
            end
        end

    end ; { Flush_buffer }


  procedure Close_output_MSX ;


    var

      rows,
      line_count,
      count : integer ;

    begin


      with cdisplay_info do
        begin
          if (msx_type = mscreen)
            then
              { pad screen files to full screen }
              begin
                if dumped_lines < 212
                  then
                    begin
                      Flush_buffer ;
                      for line_count := dumped_lines to 212 do
                        for count := 1 to msx_bytes do
                          pixel_write(0) ;
                    end
              end
            else { msx_type = mcopy }
              { pad MSX files to promised nr of lines }
              begin
                if (dumped_lines < (end_view_row - start_view_row + 1))
                  then
                    begin
                      if pixels_written < pixels_to_write
                        then
                          for count := pixels_written to
                              pixels_to_write - 1 do
                            add_msx_copy(0) ;
                      for line_count := dumped_lines to
                          (end_view_row - start_view_row + 1) do
                        for count := 1 to pixels_to_write do
                          add_msx_copy(0) ;
                    end ;
              end ;
        end ;

    end ; { Close_output_MSX }


  procedure Add_pixel_MSX(x,y,color : integer) ;


    {
     add current pixel to output device selected
     x,y coordinates of original picture
       (1..max_parse_col) (1..max_parse_row)
     cdisplay_info.current_col,row
     coordinates in output picture
       (1..(end_view_row - start_view_row + 1)
     within boundaries given by cdisplay_info.start, end row, col

    }


    procedure Add_MSX_copy_pixel ;

      {
        MSX copy files are a pixel stream, no end-of-line breaks
        if packed format (scr5-8) then packing continues over
        scanlines
        No use of buffering is made
      }

      var

        count : integer ;

      begin

        { check if new scan line entered }
        if y > previous_row
          then
            begin
              { write pixels to output to fill up line }
              if pixels_written < pixels_to_write
                then
                  for count := pixels_written to pixels_to_write - 1  do
                    add_msx_copy(0) ;
              pixels_written := 0 ;
              inc(previous_row) ;
              inc(dumped_lines) ;
            end ;

        add_msx_copy(color) ;
        inc(pixels_written) ;

      end ; { add_MSX_copy_pixel }

    procedure Add_MSX_screen_pixel ;

      var

        b : byte ;

      procedure add_12 ;

        { for screen 12 4 pixels must be packed to 4 bytes }

        var

          ra, ba, ga, ai : real ;
          ya,
          min,
          ira, iba, iga,
          b1, b2, b3, b4,
          av, cc : integer ;
          yn : array[0..3] of integer ;

        begin

          b := (x - 1) mod 4 ;
          { place RGB color in buffer and make value 0..31 }
          scr12c_RGB[b, c_red  ] := msxcolormap[color].colors[c_red  ] shr 3 ;
          scr12c_RGB[b, c_blue ] := msxcolormap[color].colors[c_blue ] shr 3 ;
          scr12c_RGB[b, c_green] := msxcolormap[color].colors[c_green] shr 3 ;
          { if all 4 pixels filled start packing }
          if b = 3
            then
              begin
                { determine base RGB color }
                ra := 0 ;
                ba := 0 ;
                ga := 0 ;
                for cc := 0 to 3 do
                  begin
                    ra := ra + scr12c_RGB[cc,c_red  ] ;
                    ba := ba + scr12c_RGB[cc,c_blue ] ;
                    ga := ga + scr12c_RGB[cc,c_green] ;
                  end ;
                ira := round(ra / 4) ;
                iba := round(ba / 4) ;
                iga := round(ga / 4) ;
                ya := 0 ;
                { determine minimum and correct }
                min := 256 ;
                if min > ira
                  then
                    min := ira ;
                if min > iga
                  then
                    min := iga ;
                if min > iba
                  then
                    min := iba ;
                if min > 0
                  then
                    min := min - 1 ;
                { shift to r or g or b = 0 }
                ira := ira - min ;
                iga := iga - min ;
                iba := iba - min ;
                { intensity of base color }
                ai := ira + iga + iba ;
                { fill in color }
                b1 := iga and 7 ;
                b2 := iga shr 3 ;
                b3 := ira and 7 ;
                b4 := ira shr 3 ;
                ya := 0 ;
(*
                if iba > 0
                  then
                    begin
                      { if J = red/blue to use }
                      if ira = 0
                        then
                          begin
                            { see if blue fits in J }
                            if (iba * 2) > 31
                              { no fit, add rest to intensity }
                              then
                                begin
                                  { B = 5y/4 + invers(j/2) }
                                  ya := round( (iba - 15) * 0.8) ;
                                  if ya < 0
                                    then ya := 0 ;
                                  iga := iga - ya ;
                                  if iga < 0
                                    then
                                      iga := 0 ;
                                  iba := 31 ;
                                end
                              else
                                iba := iba * 2 ;
                            { make inverse }
                            iba := 31 - iba ;
                            b3 := iba and 7 ;
                            b4 := (iba shr 3) + 4
                         end
                      else { must be iga = 0 }
                          begin
                            { see if blue fits in K }
                            if (iba * 4) > 31
                              { no fit, add rest to intensity }
                              then
                                begin
                                  { B = 5y/4 + invers(k/4) }
                                  ya := round( (iba - 7) * 0.8) ;
                                  if ya < 0
                                    then ya := 0 ;
                                  ira := ira - ya ;
                                  if ira < 0
                                    then
                                      ira := 0 ;
                                  iba := 31 ;
                                end
                              else
                                iba := iba * 4 ;
                            { make inverse }
                            iba := 31 - iba ;
                            b1 := iba and 7 ;
                            b2 := (iba shr 3) + 4
                          end
                    end ;
 *)
                if iga = 0
                  then
                    begin
                      b1 := 7 ;
                      b2 := 7 ;
                    end ;
                if ira = 0
                  then
                    begin
                      b3 := 7 ;
                      b4 := 7 ;
                    end ;
                { find intensity }
                for cc := 0 to 3 do
                  begin
                    yn[cc] :=
                              round((scr12c_rgb[cc, c_red  ] +
                                     scr12c_rgb[cc, c_green] +
                                     scr12c_rgb[cc, c_blue ] -  ai) / 3.0
                                     - ya ) ;
                    if yn[cc] < 0
                      then
                        yn[cc] := 0
                    else if yn[cc] > 31
                      then
                        yn[cc] := 31 ;
                  end ;
                { fill in output buffer }
                output_buffer[x -3] := b1 + (yn[0] shl 3) ;
                output_buffer[x -2] := b2 + (yn[1] shl 3) ;
                output_buffer[x -1] := b3 + (yn[2] shl 3) ;
                output_buffer[x   ] := b4 + (yn[3] shl 3) ;
              end ;

        end ; { add_12 }

      begin

      { check if new scan line entered }
      if y > previous_row
        then
          begin
            Flush_buffer ;
            inc(previous_row) ;
          end ;
      { add pixel value to output buffer, MSX screen dependent }
      case cdisplay_info.msx_screen of
        scr5, scr7 :  begin
                        if odd(x)
                          then
                            output_buffer[(x div 2) + 1] :=
                              (color shl 4) or (output_buffer[(x div 2) + 1])
                          else
                            output_buffer[(x div 2)] :=
                              (color) or (output_buffer[(x div 2)])
                      end ;
        scr6, stp6 :  begin
                        b := x mod 4 ;
                        case b of
                          1 : output_buffer[(x + 3) div 4] := color shl 6 ;
                          2 : output_buffer[(x + 3) div 4] :=
                                output_buffer[(x + 3) div 4] or
                                 (color shl 4) ;
                          3 : output_buffer[(x + 3) div 4] :=
                                output_buffer[(x + 3) div 4] or
                                 (color shl 2) ;
                          0 : output_buffer[(x + 3) div 4] :=
                                output_buffer[(x + 3) div 4] or color ;
                          end ;
                      end ;
        scr8       :  output_buffer[x] := color ;
        scr12      :  add_12 ;
      end ;

    end ; { add_MSX_copy_pixel }

  begin   { Add_pixel_MSX }

    { make MSX color through remap }
    color := remap[color] ;
    if cdisplay_info.msx_type = mscreen
      then
        Add_MSX_screen_pixel
      else
        Add_MSX_copy_pixel ;


  end ; { Add_pixel_MSX }

end. { unit smsxcr }
