/*
 *	Logging routines
 */

#ifndef __LOG_H__
#define __LOG_H__

#include <stdarg.h>
#include "16bit.h"
#include "v9t9_types.h"
#include "centry.h"

#ifdef __LOG__
static char *log_src[] =
{
	"",
	"OS",
	"Timer",
	"Parser",
	"CPU",
	"Memory",
	"CRU",
	"ROM",
	"Video",
	"Sprites",
	"Sound",
	"Keyboard",
	"Speech",

	"RealDsk",
	"EmuDsk",
	"RS232",

	"Modules",		
	"Internal",
	"Demo"
};
#endif

enum
{
	LOG_GENERAL	= 0,
	LOG_HOSTOS,
	LOG_TIMER,
	LOG_COMMANDS,
	LOG_CPU,
	LOG_MEMORY,
	LOG_CRU,
	LOG_ROMS,
	LOG_VIDEO,
	LOG_SPRITES,
	LOG_SOUND,
	LOG_KEYBOARD,
	LOG_SPEECH,

	LOG_REALDISK,
	LOG_EMUDISK,
	LOG_RS232,

	LOG_MODULES,	// v9t9 modules, not roms
	LOG_INTERNAL,	// assertion failures
	LOG_DEMO,		// demo

	LOG_NUM_SRC,

	LOG_SRC_MASK = 0x1f,
	LOG_SRC_SHIFT = 0,

	/////

	LOG_USER	= 0x100,		// user message, force visible

	// levels

	L_0			= 0 << 16,		// always
	L_1			= 1 << 16,		// level 1
	L_2			= 2 << 16,
	L_3			= 3 << 16,
	L_4			= 4 << 16,

	LOG_VERBOSE_MASK = 15 << 16,
	LOG_VERBOSE_SHIFT = 16,

	/////

	LOG_INFO	= 0 << 29,	// no header
	LOG_WARN	= 1 << 29,	// warning:
	LOG_ERROR	= 2 << 29,	// error:
	LOG_DEBUG	= 3 << 29,	// #ifdef DEBUG:  debug:
	LOG_FATAL	= 4 << 29,	// fatal:

	LOG_TYPE_MASK = 7 << 29
};

//	Test whether we echo this to the screen or not
#define LOG_IS_VISIBLE(f) ((f & LOG_USER) || ((f & LOG_TYPE_MASK) == LOG_FATAL))

//	Initialize log
void	initlog(void);
//	Add commands
void	log_add_commands(void);
//	Terminate log
void 	termlog(void);

int		log_level(int src);
const char *log_name(int src);

//	Is logging enabled given these flags?
bool 	log_enabled(int srcflags);

//	Return source flags coerced to add LOG_USER if necessary
int		log_coerced_to_user(int srcflags);

//	Log something.  srcflags=bitmask of LOG_xxx
void	vlogger(u32 srcflags, const char *format, va_list va);

void	logger(u32 srcflags, const char *format, ...);

//	Print an OS error
void	OSerror(int err, char *format, ...);

//	Definitions for various status items for the UI
//
//	The comments in (...) indicate the varargs provided to report_status().
typedef enum
{
	STATUS_CYCLES_SECOND,		// execution speed:
								// avg cycles per second (#),
								// (#) average insts per second
	STATUS_FRAMES_SECOND,		// video update frames per second (#)
	STATUS_DISK_ACCESS,			// disk DSR active (disk # 1-xx, bool on/off)
	STATUS_RS232_ACCESS,		// RS232 DSR active (port # 1-xx, bool on/off)

	// these are only reported when ST_DEBUG is set in stateflag
	STATUS_DEBUG_REFRESH,		// for each cycle
	STATUS_CPU_PC,				// PC value (addr)
	STATUS_CPU_STATUS,			// status value (#)
	STATUS_CPU_WP,				// WP value (addr)
	STATUS_CPU_REGISTER_VIEW,	// register view changed (wp, ptr to 16 regs)
	STATUS_CPU_REGISTER_READ,	// register read (reg#, val)
	STATUS_CPU_REGISTER_WRITE,	// register changed (reg#, val)
	STATUS_CPU_INSTRUCTION,		// instruction (Instruction *, hex, opdisasm, op1, op2)
	STATUS_CPU_INSTRUCTION_LAST,// last instruction (Instruction *, op1, op2)

	// a memory view is 'debugger_memory_view_size' bytes long
	STATUS_MEMORY_VIEW,			// memory view changes (Memory*)
	STATUS_MEMORY_READ,			// memory read in view (Memory*)
	STATUS_MEMORY_WRITE,		// memory write in view (Memory*)
}	status_item;

//	Print a status item for frontend
//	The descriptions above tell what parameters will follow.
void
report_status(status_item item, ...);

//	Format status item into text, if verbosity allows it
void
report_status_text(status_item item, va_list va, char *buffer, int bufsz);

void	my_assert_func(char *file, int line, char *message);

#define _(x)	x

void    my_assert_func(char *file, int line, char *message);

#define my_assert(x) \
do { if (!(x)) my_assert_func(__FILE__, __LINE__, #x); } while (0)

#define DEBUG

#include "cexit.h"

#endif


