/*
	Generate a version 2 hard disk image suitable for MESS drivers that use
	sector len other than 512.

	Syntax:
		[program] dest blocksize cylinders heads sectors/tracks bytes/sector
		dest: file name of dest
		blocksize: can be 1
		cylinders: # of cylinders in hard disk
		heads: # of heads in hard disk
		sectors/tracks: # of sectors per track in hard disk
		bytes/sector: sector length: 512 for ide and scsi disks, 256 for ti99
			hfdc card, usually 256 for ti990 (but other values are accepted)

	Example:
		[program] test.hd 1 615 4 32 256
		makes a 20MB image for the ti99 hfdc

	Raphael Nabet 2003
*/

/* Enable thi line if compiling for alittle-endian host */
/*#define LSB_FIRST*/

#if defined(__MWERKS__) && macintosh
#include <console.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

typedef unsigned char UINT8;
typedef unsigned long UINT32;
typedef unsigned long long UINT64;

typedef UINT64 mapentry_t;

#define HARD_DISK_HEADER_SIZE		80
#define HARD_DISK_HEADER_VERSION	2

#define HDFLAGS_IS_WRITEABLE		0x00000002

#define HDCOMPRESSION_NONE			0

struct hard_disk_header
{
	UINT32	length;				/* length of header data */
	UINT32	version;			/* drive format version */
	UINT32	flags;				/* flags field */
	UINT32	compression;		/* compression type */
	UINT32	blocksize;			/* sectors per block */
	UINT32	totalblocks;		/* total # of blocks represented */
	UINT32	cylinders;			/* number of cylinders on hard disk */
	UINT32	heads;				/* number of heads on hard disk */
	UINT32	sectors;			/* number of sectors on hard disk */
	UINT32	seclen;
	UINT8	md5[16];			/* MD5 checksum for this drive */
	UINT8	parentmd5[16];		/* MD5 checksum for parent drive */
};

#define MAX_SECLEN 2048

#define END_OF_LIST_COOKIE	"EndOfListCookie"
#define SET_ERROR_AND_CLEANUP(err) do { goto cleanup; } while (0)
#define INLINE inline

INLINE void encode_mapentry(mapentry_t *entry, UINT64 offset, UINT32 length)
{
	*entry = ((UINT64)length << 44) | ((offset << 20) >> 20);
}

INLINE void put_bigendian_uint32(UINT8 *base, UINT32 value)
{
	base[0] = value >> 24;
	base[1] = value >> 16;
	base[2] = value >> 8;
	base[3] = value;
}

INLINE void put_mapentry(UINT8 *base, mapentry_t value)
{
	base[0] = value >> 56;
	base[1] = value >> 48;
	base[2] = value >> 40;
	base[3] = value >> 32;
	base[4] = value >> 24;
	base[5] = value >> 16;
	base[6] = value >> 8;
	base[7] = value;
}

static size_t fseekwrite(FILE *stream, UINT64 offset, UINT32 count, const void *buffer)
{
	if (fseek(stream, offset, SEEK_SET))
		return 0;
	else
		return fwrite(buffer, 1, count, stream);
}


static int write_header(FILE *stream, const struct hard_disk_header *header)
{
	UINT8 rawheader[HARD_DISK_HEADER_SIZE];
	UINT32 count;

	/* assemble the data */
	memset(rawheader, 0, sizeof(rawheader));
	memcpy(rawheader, "MComprHD", 8);
	put_bigendian_uint32(&rawheader[8],  HARD_DISK_HEADER_SIZE);
	put_bigendian_uint32(&rawheader[12], header->version);
	put_bigendian_uint32(&rawheader[16], header->flags);
	put_bigendian_uint32(&rawheader[20], header->compression);
	put_bigendian_uint32(&rawheader[24], header->blocksize);
	put_bigendian_uint32(&rawheader[28], header->totalblocks);
	put_bigendian_uint32(&rawheader[32], header->cylinders);
	put_bigendian_uint32(&rawheader[36], header->heads);
	put_bigendian_uint32(&rawheader[40], header->sectors);
	memcpy(&rawheader[44], header->md5, 16);
	memcpy(&rawheader[60], header->parentmd5, 16);
	put_bigendian_uint32(&rawheader[76], header->seclen);

	/* seek and write */
	count = fseekwrite(stream, 0, sizeof(rawheader), rawheader);
	if (count != sizeof(rawheader))
		return 1;

	return 0;
}

int main(int argc, char *argv[])
{
	char *filename;
	int blocksize, cylinders, heads, sectors, seclen;
	UINT8 blanksector[MAX_SECLEN];
	UINT8 rawentry[8];
	mapentry_t current_entry;
	int count;
	struct hard_disk_header header;
	UINT64 fileoffset, entryoffset;
	FILE *stream = NULL;
	int i, j, err;


#if defined(__MWERKS__) && macintosh
	argc = ccommand(& argv);
#endif

	if (argc != 7)
	{
		fprintf(stderr, "%s dest blocksize cylinders heads sectors/tracks bytes/sector\n");
		goto cleanup;
	}

	filename = argv[1];
	blocksize = atoi(argv[2]);
	cylinders = atoi(argv[3]);
	heads = atoi(argv[4]);
	sectors = atoi(argv[5]);
	seclen = atoi(argv[6]);

	/* sanity check the header, filling in missing info */
 	header.length = HARD_DISK_HEADER_SIZE;
	header.version = HARD_DISK_HEADER_VERSION;
	header.flags = HDFLAGS_IS_WRITEABLE;
	header.compression = HDCOMPRESSION_NONE;
	header.blocksize = blocksize;
	header.cylinders = cylinders;
	header.heads = heads;
	header.sectors = sectors;
	header.seclen = seclen;

	/* require a valid blocksize */
	if (header.blocksize == 0 || header.blocksize >= 2048)
	{
		fprintf(stderr, "invalid blocksize\n");
		goto cleanup;
	}

	/* require either totalsectors or CHS values */
	if (!header.cylinders || !header.sectors || !header.heads)
	{
		fprintf(stderr, "invalid geometry\n");
		goto cleanup;
	}

	/* derive the remaining data */
	header.totalblocks = ((header.cylinders * header.sectors * header.heads) + header.blocksize - 1) / header.blocksize;

	/* clear checksums */
	memset(header.md5, 0, 16);
	memset(header.parentmd5, 0, 16);

	/* attempt to create the file */
	stream = fopen(filename, "wb");
	if (!stream)
	{
		fprintf(stderr, "can't open dest\n");
		goto cleanup;
	}

	/* write the resulting header */
	err = write_header(stream, &header);
	if (err)
	{
		fprintf(stderr, "write error\n");
		goto cleanup;
	}

	/* prepare to write a flat sector map immediately following */
	fileoffset = header.length;
	entryoffset = header.length + header.totalblocks*sizeof(current_entry) + sizeof(current_entry);

	/* write map entries */
	for (i = 0; i < header.totalblocks; i++)
	{
		encode_mapentry(& current_entry, entryoffset, header.blocksize*header.seclen);
		put_mapentry(rawentry, current_entry);
		count = fseekwrite(stream, fileoffset, 8, rawentry);
		if (count != 8)
		{
			fprintf(stderr, "write error\n");
			goto cleanup;
		}
		fileoffset += sizeof(current_entry);
		entryoffset += header.blocksize*header.seclen;
	}

	/* then write a special end-of-list cookie */
	memcpy(& current_entry, END_OF_LIST_COOKIE, sizeof(current_entry));
	count = fseekwrite(stream, fileoffset, sizeof(current_entry), & current_entry);
	if (count != sizeof(current_entry))
	{
		fprintf(stderr, "write error\n");
		goto cleanup;
	}
	fileoffset += sizeof(current_entry);

	/* then write the data blocks */
	memset(blanksector, 0, header.seclen);
	for (i = 0; i < header.totalblocks; i++)
	{
		for (j = 0; j < header.blocksize; j++)
		{
			count = fseekwrite(stream, fileoffset, header.seclen, blanksector);
			if (count != header.seclen)
			{
				fprintf(stderr, "write error\n");
				goto cleanup;
			}
			fileoffset += header.seclen;
		}
	}

	/* all done */
	fclose(stream);
	return 0;

cleanup:
	if (stream)
		fclose(stream);
	return 1;
}